// ============================================================
// 🚀 FILE: assets/js/mega-menu-fixes.js
// PURPOSE: Critical fixes for Super Mega Menu sidebar height and mobile menu visibility
// ============================================================

document.addEventListener('DOMContentLoaded', function() {
    console.log('Mega Menu Fixes: Initializing'); // Debug: Confirm initialization
    
    // ============================================================
    // 🧩 Section: Apply Critical CSS Fixes
    // PURPOSE: Apply critical CSS fixes directly to ensure they take effect
    // ============================================================
    function applyCriticalCSSFixes() {
        // Create a style element
        const style = document.createElement('style');
        style.type = 'text/css';
        style.id = 'mega-menu-critical-fixes';
        
        // Add the critical CSS fixes
        style.innerHTML = `
            /* CRITICAL FIX: Ensure sidebar extends full height in desktop mode */
            .side_bar {
                position: fixed !important;
                top: 0 !important;
                right: 0 !important;
                bottom: 0 !important;
                height: 100vh !important; /* Force full viewport height */
                width: 320px !important; /* Standard width */
                max-width: 90% !important; /* Responsive on smaller screens */
                background-color: #ffffff !important; /* Background Color: #ffffff (White) */
                color: #333333 !important; /* Text Color: #333333 (Dark Gray) */
                z-index: 99999 !important; /* Very high z-index to ensure visibility */
                overflow-y: auto !important; /* Allow scrolling */
                box-shadow: -2px 0 10px rgba(0, 0, 0, 0.2) !important; /* Add shadow for depth */
                transition: transform 0.3s ease-in-out !important;
                transform: translateX(100%) !important; /* Start off-screen */
                padding: 20px !important;
                display: flex !important;
                flex-direction: column !important;
            }

            /* When sidebar is visible */
            .side_bar.slideInRight {
                transform: translateX(0) !important; /* Slide in from right */
                display: flex !important;
            }

            /* When sidebar is hidden */
            .side_bar.side_bar_hidden {
                transform: translateX(100%) !important; /* Move off-screen */
                display: none !important;
            }

            /* CRITICAL FIX: Ensure mobile mega menu is visible in Pages dropdown */
            #targetElement .accordion-toggle,
            .mobile-menu .accordion-toggle {
                background-color: #407a5a !important; /* BG Color: #407a5a (Primary Green) */
                color: #ffffff !important; /* Text Color: #ffffff (White) */
                padding: 10px 15px !important;
                margin: 5px 0 !important;
                border-radius: 4px !important;
                display: block !important;
                visibility: visible !important;
                opacity: 1 !important;
                cursor: pointer !important;
                font-weight: bold !important;
                text-align: left !important;
            }

            /* CRITICAL FIX: Ensure mobile mega menu content is visible */
            #targetElement .accordion-content,
            .mobile-menu .accordion-content {
                background-color: #f2f2f2 !important; /* BG Color: #f2f2f2 (Panel Gray) */
                color: #333333 !important; /* Text Color: #333333 (Dark Gray) */
                padding: 10px !important;
                margin: 0 0 10px 0 !important;
                border-radius: 0 0 4px 4px !important;
                border-left: 3px solid #407a5a !important; /* Border Color: #407a5a (Primary Green) */
                display: none !important; /* Hidden by default */
            }

            /* CRITICAL FIX: Ensure mobile mega menu content is visible when active */
            #targetElement .accordion-content.active,
            .mobile-menu .accordion-content.active {
                display: block !important;
                visibility: visible !important;
                opacity: 1 !important;
            }

            /* CRITICAL FIX: Ensure all text in mobile menu is visible */
            #targetElement *,
            .mobile-menu * {
                color: #333333 !important; /* Text Color: #333333 (Dark Gray) */
                text-shadow: none !important;
            }

            /* CRITICAL FIX: Ensure all links in mobile menu are visible */
            #targetElement a,
            .mobile-menu a {
                color: #407a5a !important; /* Link Color: #407a5a (Primary Green) */
                text-decoration: none !important;
                display: block !important;
                padding: 5px 0 !important;
                margin: 2px 0 !important;
                visibility: visible !important;
                opacity: 1 !important;
            }

            /* CRITICAL FIX: Ensure all headings in mobile menu are visible */
            #targetElement h3,
            #targetElement h4,
            #targetElement h5,
            #targetElement h6,
            .mobile-menu h3,
            .mobile-menu h4,
            .mobile-menu h5,
            .mobile-menu h6 {
                color: #407a5a !important; /* Heading Color: #407a5a (Primary Green) */
                font-weight: bold !important;
                margin: 10px 0 5px 0 !important;
                visibility: visible !important;
                opacity: 1 !important;
            }

            /* CRITICAL FIX: Ensure mobile menu is properly positioned */
            @media (max-width: 991.98px) {
                /* Mobile mega menu container */
                .mega-menu-container.active {
                    position: fixed !important;
                    top: 60px !important; /* Position below header */
                    left: 0 !important;
                    right: 0 !important;
                    width: 100% !important;
                    height: auto !important;
                    max-height: calc(100vh - 60px) !important; /* Limit height to avoid banner overlap */
                    border-radius: 0 !important;
                    z-index: 99990 !important;
                    transform: none !important;
                    overflow-y: auto !important;
                    background-color: #ffffff !important; /* Background Color: #ffffff (White) */
                    padding: 15px !important;
                    display: block !important;
                    visibility: visible !important;
                    opacity: 1 !important;
                }
                
                /* Ensure sidebar is full height but doesn't extend beyond viewport */
                .side_bar {
                    height: 100vh !important;
                    max-height: 100vh !important;
                    overflow-y: auto !important;
                    width: 80% !important;
                    max-width: 320px !important;
                }
            }
        `;
        
        // Add the style element to the head
        document.head.appendChild(style);
        
        console.log('Mega Menu Fixes: Applied critical CSS fixes'); // Debug: Log CSS application
    }
    
    // ============================================================
    // 🧩 Section: Fix Sidebar Height
    // PURPOSE: Ensure sidebar extends full height in desktop mode
    // ============================================================
    function fixSidebarHeight() {
        const sidebar = document.querySelector('#targetElement');
        if (!sidebar) {
            console.warn('Mega Menu Fixes: Sidebar (#targetElement) not found'); // Debug: Warn if missing
            return;
        }
        
        // Apply critical styles to ensure full height
        sidebar.style.position = 'fixed';
        sidebar.style.top = '0';
        sidebar.style.right = '0';
        sidebar.style.bottom = '0';
        sidebar.style.height = '100vh';
        sidebar.style.maxHeight = '100vh';
        sidebar.style.width = window.innerWidth <= 768 ? '80%' : '320px';
        sidebar.style.maxWidth = '320px';
        sidebar.style.backgroundColor = '#ffffff';
        sidebar.style.color = '#333333';
        sidebar.style.zIndex = '99999';
        sidebar.style.overflowY = 'auto';
        sidebar.style.boxShadow = '-2px 0 10px rgba(0, 0, 0, 0.2)';
        sidebar.style.transition = 'transform 0.3s ease-in-out';
        sidebar.style.padding = '20px';
        sidebar.style.display = sidebar.classList.contains('side_bar_hidden') ? 'none' : 'flex';
        sidebar.style.flexDirection = 'column';
        
        // Ensure transform is properly set based on visibility
        if (sidebar.classList.contains('slideInRight')) {
            sidebar.style.transform = 'translateX(0)';
        } else {
            sidebar.style.transform = 'translateX(100%)';
        }
        
        console.log('Mega Menu Fixes: Applied full height styles to sidebar'); // Debug: Log style application
    }
    
    // ============================================================
    // 🧩 Section: Fix Mobile Menu Visibility
    // PURPOSE: Ensure mobile menu items are visible with proper contrast
    // ============================================================
    function fixMobileMenuVisibility() {
        // Target all accordion toggles in the mobile menu
        const accordionToggles = document.querySelectorAll('#targetElement .accordion-toggle, .mobile-menu .accordion-toggle');
        accordionToggles.forEach(toggle => {
            toggle.style.backgroundColor = '#407a5a';
            toggle.style.color = '#ffffff';
            toggle.style.padding = '10px 15px';
            toggle.style.margin = '5px 0';
            toggle.style.borderRadius = '4px';
            toggle.style.display = 'block';
            toggle.style.visibility = 'visible';
            toggle.style.opacity = '1';
            toggle.style.cursor = 'pointer';
            toggle.style.fontWeight = 'bold';
            toggle.style.textAlign = 'left';
        });
        
        // Target all accordion content in the mobile menu
        const accordionContents = document.querySelectorAll('#targetElement .accordion-content, .mobile-menu .accordion-content');
        accordionContents.forEach(content => {
            content.style.backgroundColor = '#f2f2f2';
            content.style.color = '#333333';
            content.style.padding = '10px';
            content.style.margin = '0 0 10px 0';
            content.style.borderRadius = '0 0 4px 4px';
            content.style.borderLeft = '3px solid #407a5a';
            
            // Only display if active
            if (content.classList.contains('active')) {
                content.style.display = 'block';
                content.style.visibility = 'visible';
                content.style.opacity = '1';
            } else {
                content.style.display = 'none';
            }
        });
        
        // Target all text elements in the mobile menu
        const textElements = document.querySelectorAll('#targetElement *, .mobile-menu *');
        textElements.forEach(el => {
            // Skip elements that don't have style property
            if (!el.style) return;
            
            // Skip accordion toggles (already styled above)
            if (el.classList.contains('accordion-toggle')) return;
            
            // Skip accordion content (already styled above)
            if (el.classList.contains('accordion-content')) return;
            
            // Apply text color
            el.style.color = '#333333';
            el.style.textShadow = 'none';
        });
        
        // Target all links in the mobile menu
        const links = document.querySelectorAll('#targetElement a, .mobile-menu a');
        links.forEach(link => {
            link.style.color = '#407a5a';
            link.style.textDecoration = 'none';
            link.style.display = 'block';
            link.style.padding = '5px 0';
            link.style.margin = '2px 0';
            link.style.visibility = 'visible';
            link.style.opacity = '1';
        });
        
        // Target all headings in the mobile menu
        const headings = document.querySelectorAll('#targetElement h3, #targetElement h4, #targetElement h5, #targetElement h6, .mobile-menu h3, .mobile-menu h4, .mobile-menu h5, .mobile-menu h6');
        headings.forEach(heading => {
            heading.style.color = '#407a5a';
            heading.style.fontWeight = 'bold';
            heading.style.margin = '10px 0 5px 0';
            heading.style.visibility = 'visible';
            heading.style.opacity = '1';
        });
        
        console.log('Mega Menu Fixes: Applied visibility styles to mobile menu'); // Debug: Log style application
    }
    
    // ============================================================
    // 🧩 Section: Fix Mega Menu Dropdowns in Mobile Mode
    // PURPOSE: Ensure mega menu dropdowns are visible in mobile mode
    // ============================================================
    function fixMegaMenuDropdowns() {
        // Target the vertical and horizontal mega menus
        const verticalMenu = document.querySelector('#vertical-mega-menu-home');
        const horizontalMenu = document.querySelector('#horizontal-mega-menu-pages');
        
        if (verticalMenu) {
            // Ensure the vertical menu is properly styled
            verticalMenu.style.backgroundColor = '#ffffff';
            verticalMenu.style.color = '#333333';
            verticalMenu.style.zIndex = '20001';
            
            // Ensure all text elements in the vertical menu are visible
            const verticalMenuElements = verticalMenu.querySelectorAll('*');
            verticalMenuElements.forEach(el => {
                // Skip elements that don't have style property
                if (!el.style) return;
                
                // Apply text color
                el.style.color = '#333333';
                el.style.textShadow = 'none';
            });
            
            // Ensure all links in the vertical menu are visible
            const verticalMenuLinks = verticalMenu.querySelectorAll('a');
            verticalMenuLinks.forEach(link => {
                link.style.color = '#407a5a';
                link.style.textDecoration = 'none';
                link.style.display = 'block';
                link.style.padding = '5px 0';
                link.style.margin = '2px 0';
                link.style.visibility = 'visible';
                link.style.opacity = '1';
            });
            
            console.log('Mega Menu Fixes: Applied visibility styles to vertical mega menu'); // Debug: Log style application
        } else {
            console.warn('Mega Menu Fixes: Vertical mega menu (#vertical-mega-menu-home) not found'); // Debug: Warn if missing
        }
        
        if (horizontalMenu) {
            // Ensure the horizontal menu is properly styled
            horizontalMenu.style.backgroundColor = '#ffffff';
            horizontalMenu.style.color = '#333333';
            horizontalMenu.style.zIndex = '20001';
            
            // Ensure all text elements in the horizontal menu are visible
            const horizontalMenuElements = horizontalMenu.querySelectorAll('*');
            horizontalMenuElements.forEach(el => {
                // Skip elements that don't have style property
                if (!el.style) return;
                
                // Apply text color
                el.style.color = '#333333';
                el.style.textShadow = 'none';
            });
            
            // Ensure all links in the horizontal menu are visible
            const horizontalMenuLinks = horizontalMenu.querySelectorAll('a');
            horizontalMenuLinks.forEach(link => {
                link.style.color = '#407a5a';
                link.style.textDecoration = 'none';
                link.style.display = 'block';
                link.style.padding = '5px 0';
                link.style.margin = '2px 0';
                link.style.visibility = 'visible';
                link.style.opacity = '1';
            });
            
            console.log('Mega Menu Fixes: Applied visibility styles to horizontal mega menu'); // Debug: Log style application
        } else {
            console.warn('Mega Menu Fixes: Horizontal mega menu (#horizontal-mega-menu-pages) not found'); // Debug: Warn if missing
        }
    }
    
    // Apply all fixes
    applyCriticalCSSFixes();
    fixSidebarHeight();
    fixMobileMenuVisibility();
    fixMegaMenuDropdowns();
    
    // Apply fixes when the sidebar is opened
    const openButton = document.querySelector('#openButton');
    if (openButton) {
        openButton.addEventListener('click', function() {
            // Wait a short time for the sidebar to be added to the DOM
            setTimeout(function() {
                fixSidebarHeight();
                fixMobileMenuVisibility();
            }, 100);
        });
    }
    
    // Apply fixes when the window is resized
    window.addEventListener('resize', function() {
        fixSidebarHeight();
        fixMobileMenuVisibility();
        fixMegaMenuDropdowns();
    });
    
    // Apply fixes when dropdown menus are clicked
    const dropdownElements = document.querySelectorAll('.dropdown-hover');
    dropdownElements.forEach(dropdown => {
        dropdown.addEventListener('click', function() {
            // Wait a short time for the menu to be displayed
            setTimeout(function() {
                fixMegaMenuDropdowns();
            }, 100);
        });
    });
    
    console.log('Mega Menu Fixes: All fixes applied'); // Debug: Confirm all fixes applied
});